﻿using Microscopic_Traffic_Simulator.Renderers;
using System.Collections.Generic;
using System.Windows;

namespace Microscopic_Traffic_Simulator.Views.MainWindowView
{
    /// <summary>
    /// Contains actions related to zooming.
    /// </summary>
    class CanvasZoomer
    {
        /// <summary>
        /// List used as mapping of zoom setting to zoom rate in pixels per meter.
        /// </summary>
        private List<double> zoomRates = new List<double>()
        {
          0.01,0.010185107422272,0.0107462594300664,0.0116924917303975,0.0130332406116991,0.0147783736935546,
          0.0169382222242854,0.0195235956656836,0.0225458459697171,0.0260168794200501,0.0299492050471733,
          0.034356001790748,0.0392511198356065,0.0446492040091223,0.0505657079744171,0.0570169547790742,
          0.0640202635608923,0.0715939990762884,0.0797576563313233,0.0885320456061308,0.0979392961362084,
          0.108003098526732,0.118748782237848,0.130203609399974,0.142396805696897,0.155359994146626,
          0.169127324722157,0.183735870145639,0.199225972795153,0.215641710029827,0.233031425276786,0.25144828898761,
          0.27095116916386,0.291605455836101,0.313484368635317,0.336670101334579,0.361255856987923,0.387348080465699,
          0.415069320821721,0.444562290104624,0.475994978437067,0.5095682999324,0.545526216301859,0.584171301578586,
          0.625888740355871,0.671184831072584,0.720755181737304,0.775616555528577,0.837402218209764,0.90921485771794,1,
          1.11983649303588,1.2915491581914,1.50246505421121,1.74822366593979,2.02661042175123,2.33638811921807,
          2.67687534154144,3.0477492516073,3.44894368602695,3.8805916710927,4.34298534243674,4.83655678339864,
          5.36186735366406,5.91959163361002,6.51052322033564,7.13556944119942,7.79575291651485,8.49221854628459,
          9.22624342549964,9.99923769484131,10.8127664104111,11.6685557709354,12.5685116489888,13.5147472321006,
          14.5095982549991,15.5556620326794,16.6558201304091,17.813299802602,19.0317065734665,20.3151047855808,
          21.6680840663906,23.0958606964767,24.6044065590172,26.2005938080285,27.8923992747045,29.6891756054087,
          31.6019765044516,33.6440516341236,35.8314830594365,38.1841152325796,40.7269171567223,43.4920221803057,
          46.5220093353223,49.8753604917599,53.6362584385355,57.9338841330926,62.9856673317946,69.2154567347481,
          77.7170870631988,100
        };

        /// <summary>
        /// List of the renderers on the canvas.
        /// </summary>
        private List<VisualCanvasRenderer> renderers;        

        /// <summary>
        /// Canvas location whose real-world location should stay on the same place.
        /// </summary>
        private Point? zoomPoint = null;

        /// <summary>
        /// Initialize the canvas zoomer.
        /// </summary>
        /// <param name="renderers">List of the renderers on a canvas.</param>
        public CanvasZoomer(List<VisualCanvasRenderer> renderers)
        {            
            this.renderers = renderers;            
        }

        /// <summary>
        /// Applies the new zoom rate and redraw the canvas.
        /// </summary>
        /// <param name="zoomSetting">New zoom setting.</param>
        /// <returns>Value of the zoom rate in pixels per meter.</returns>
        internal double UpdatePixelsPerMeter(int zoomSetting, double canvasWidth, double canvasHeight)
        {
            if (!zoomPoint.HasValue)
            {
                zoomPoint = new Point(canvasWidth / 2.0, canvasHeight / 2.0);
            }
            double newZoomRate = zoomRates[zoomSetting];
            renderers.ForEach(i => i.ChangeZoom(newZoomRate, zoomPoint.Value));
            zoomPoint = null;
            return newZoomRate;
        }


        /// <summary>
        /// Sets point whose location should be same after zoom change.
        /// </summary>
        /// <param name="point">Zoom point.</param>
        internal void SetCurrentCursorPositionAsZoomPoint(Point point)
        {
            zoomPoint = point;
        }
    }
}
